//*************************************************************************************************
//
//	Description:
//		PostProcessing.fxh
//
//	<P> Copyright (c) 2007 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Alastair Murray
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		AMurray		    01/10/2007  0.2           Use a Define rather than Static Branching
//		AMurray		    26/09/2007  0.1           Created
//	<TABLE>
//
//*************************************************************************************************


// NOTE: Use the defines POST_PROCESS_EFFECT_INDEX, POST_PROCESS_EFFECT_INDEX2, POST_PROCESS_EFFECT_INDEX3 
// to determine the index of the effects applied.
// e.g. #define POST_PROCESS_EFFECT_INDEX	GREYSCALE

#define	NONE														0
#define	OLD_TV													1
#define	GREYSCALE												2
#define	SEPIA														3
#define	COLOURISED											4
#define	SATURATE_COLOURISE							5
#define	CONTRAST_INCREASE								6
#define	REMOVE_COLOUR										7
#define	SATURATE												8
#define	MOSAIC													9
#define	MULTIPLY_RGB										10
#define	TV_FRAME												11
#define	TV_FLICKER											12
#define	TV_INTERFERENCE									13
#define	TV_BLOTCHES											14
#define	TV_FLICKER_AND_INTERFERENCE			15
#define	BLUR1														16
#define	BLUR2														17
#define	GREYSCALE_RGB										18
#define	BLUR1_AND_GREYSCALE							19
#define	BLUR1_AND_GREYSCALE_AND_CONTRAST_INCREASE		20
#define	IMPACT													21

#if !defined(_PS3_)

// mark certain combinations as using alphablend (i.e. they don't need to capture and blend with the frame buffer)
#if ( POST_PROCESS_EFFECT_INDEX==TV_FRAME ) && ( POST_PROCESS_EFFECT_INDEX2==NONE ) && ( POST_PROCESS_EFFECT_INDEX3==NONE )
	#define USE_ALPHA_BLEND_ZERO_SRCCOLOR
#elif ( POST_PROCESS_EFFECT_INDEX==NONE ) && ( POST_PROCESS_EFFECT_INDEX2==NONE ) && ( POST_PROCESS_EFFECT_INDEX3==NONE )
	#define USE_ALPHA_BLEND_ZERO_SRCCOLOR
	bool EmptyFilter;	// fake param to inform the code that this filter can be skipped in some cases
#elif ( POST_PROCESS_EFFECT_INDEX==MULTIPLY_RGB ) && ( POST_PROCESS_EFFECT_INDEX2==NONE ) && ( POST_PROCESS_EFFECT_INDEX3==NONE )
	#define USE_ALPHA_BLEND_ZERO_SRCCOLOR
#endif

#ifdef USE_ALPHA_BLEND_ZERO_SRCCOLOR
bool UsesAlphaBlend;	// this is a fake param used to tell the code that alpha blend needs to be on
#endif

#endif

#ifdef _PS3_
	#ifdef _POST_PARSE_
			#define	SCENE_TEX2D( _uv )	f4tex2D( Map4X, (_uv) )
	#else
			#define	SCENE_TEX2D( _uv )	tex2D( Map4X, (_uv) )
	#endif
#else
	#define	SCENE_TEX2D( _uv )	tex2D( SceneInput, (_uv) )
#endif



float2	texinput_width_height;
float	alpha_output = 1.0f;


float3 RGB_to_HSV( float3 RGB )
{
	RGB = saturate( RGB );
    float3 HSV = 0.0f;
    float minVal = min(min(RGB.r,RGB.g),RGB.b);
    float maxVal = max(max(RGB.r,RGB.g),RGB.b);
    float delta = maxVal - minVal;             //Delta RGB value 
    HSV.z = maxVal;
    if (delta != 0) 
	{                    // If gray, leave H & S at zero
       HSV.y = delta / maxVal;
       float3 delRGB;
       delRGB = ( ( ( maxVal.xxx - RGB ) / 6.0 ) + ( delta / 2.0 ) ) / delta;
       if      ( RGB.x == maxVal ) HSV.x = delRGB.z - delRGB.y;
       else if ( RGB.y == maxVal ) HSV.x = ( 1.0/3.0) + delRGB.x - delRGB.z;
       else if ( RGB.z == maxVal ) HSV.x = ( 2.0/3.0) + delRGB.y - delRGB.x;
       if ( HSV.x < 0.0 ) { HSV.x += 1.0; }
       if ( HSV.x > 1.0 ) { HSV.x -= 1.0; }
    }
    return (HSV);
}

float3 HSV_to_RGB( float3 HSV )
{
    float3 RGB = HSV.z;
    if ( HSV.y != 0 ) 
	{
       float var_h = HSV.x * 6;
       float var_i = floor(var_h);   // Or ... var_i = floor( var_h )
       float var_1 = HSV.z * (1.0 - HSV.y);
       float var_2 = HSV.z * (1.0 - HSV.y * (var_h-var_i));
       float var_3 = HSV.z * (1.0 - HSV.y * (1-(var_h-var_i)));
       if      (var_i == 0) { RGB = float3(HSV.z, var_3, var_1); }
       else if (var_i == 1) { RGB = float3(var_2, HSV.z, var_1); }
       else if (var_i == 2) { RGB = float3(var_1, HSV.z, var_3); }
       else if (var_i == 3) { RGB = float3(var_1, var_2, HSV.z); }
       else if (var_i == 4) { RGB = float3(var_3, var_1, HSV.z); }
       else                 { RGB = float3(HSV.z, var_1, var_2); }
   }
   return (RGB);
}

float	greyscale_amount = 1.0f;

float3	GreyScale( float3 c )
{
	float3	grey = dot( c.rgb, float3(0.3f, 0.59f, 0.11f) );
	c.rgb = lerp( c.rgb, grey, greyscale_amount );
	return c;
}

float3	greyscale_RGB_amount = float3(1.0f,1.0f,1.0f);

float3	GreyScaleRGB( float3 c )
{
	float3	grey = dot( c.rgb, float3(0.3f, 0.59f, 0.11f) );
	c.rgb = lerp( c.rgb, grey.rgb, greyscale_RGB_amount );
	return c;
}

float	sepia_amount = 1.0f;

float3	Sepia( float3 c )
{
	float3 LightColour = float3( 1.0f, 0.9f, 0.5f );
	float3 DarkColor = float3( 0.2f, 0.05f, 0.0f );
	float Desat = 0.5f;

	float3 scnColor = LightColour * c.rgb;
	float gray = dot( float3(0.3f, 0.59f, 0.11f),scnColor);
	float3 muted = lerp(scnColor,gray.rrr,Desat);
	float3 sepia = lerp(DarkColor,LightColour,gray);
	float3 result = lerp(muted,sepia,sepia_amount);
	return result;
}

float	colourised_amount = 1.0f;

float3	Colourised( float3 c )
{
	float3	temp = c.rgb*float3(1.0f,0.75f,1.0f);
	temp = saturate( (temp-float3(0.25f,0.0f,0.0f))*2.0f );
	temp = saturate(temp-float3(0.4f, 0.4f, 0.4f));
	temp *= 8.0f;
	c.rgb = lerp( c.rgb, temp, colourised_amount );
	return c;
}

float	saturate_colourise_amount = 1.0f;

float3	SaturateColourise( float3 c )
{
	float3	temp = saturate( ((c.rgb*2.0f)-1.0f)*4.0f );			// simple filter
	float3	temp2 = saturate(temp * float3(0.3f, 0.3f, 0.3f));		// look for bright colours
	float	gray = dot( c.rgb, float3(0.3f, 0.59f, 0.11f) );		// greyscale
	temp = lerp( temp, gray.rrr, gray );	// lrp between colour and grey
	c.rgb = lerp( c.rgb, temp, saturate_colourise_amount );
	return c;
}

float	contrastincrease_value = 0.5f;
float	contrastincrease_amount = 1.0f;

float3	ContrastIncrease( float3 c )
{
	float3	gray = dot( c.rgb, float3(0.3f, 0.59f, 0.11f) );		// greyscale
	//determine what multiplier to use based upon intensity in gray
	float3	t2 = 1.0f-gray;
	float3	r1 = (t2*3.0f)+1.0f;
	float3	r0 = gray*r1*c.rgb;	// brightener * original
	// lerp back to original use half-brightness as an indication of how far to lerp
	t2 *= contrastincrease_value;
	float3 newcolour = lerp( r0, c.rgb, 1.0f-t2 );
	c.rgb = lerp( c.rgb, newcolour, contrastincrease_amount );
	return c;
}

float	remove_colour_amount = 1.0f;

float3	RemoveColour( float3 c )
{
	float3	reduceG = c.rgb * float3(1.0f,0.75f,1.0f);
	float3	reduceR = saturate( reduceG - float3(0.25f,0.0f,0.0f) );
	float3	inv = 1.0f - reduceG;
	inv *= inv;
	inv *= inv;
	inv *= inv;
	reduceR = reduceR - inv;
	c.rgb = lerp( c.rgb, reduceR, remove_colour_amount );
	return c;
}

float	saturate_amount;

float3	Saturate( float3 c )
{
	float3	hsv = RGB_to_HSV( c );
	hsv.y *= saturate_amount;
	c.rgb = HSV_to_RGB( hsv );
	return c;
}


texture noise1DTexture : TEXTURE;
sampler noise1DInput : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="noise1DTexture"; 
	string MinFilter = "Point";
	string MagFilter = "Point";
	string MipFilter = "None";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < noise1DTexture >;
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	AddressU  = Clamp;
	AddressV  = Clamp;
	MinFilter = Point;
	MagFilter = Point;
	MipFilter = None;
	SET_NO_ANISOTROPY
};


texture noise1DTexture2 : TEXTURE;
sampler noise1DInput2 : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="noise1DTexture2"; 
	string MinFilter = "Point";
	string MagFilter = "Point";
	string MipFilter = "None";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	int MipMapLODBias = 0;
> = sampler_state
{
	Texture = < noise1DTexture2 >;
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	AddressU  = Clamp;
	AddressV  = Clamp;
	MinFilter = Point;
	MagFilter = Point;
	MipFilter = None;
	SET_NO_ANISOTROPY
};


texture TVsplotchesTexture : TEXTURE;
sampler TVsplotchesInput : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="TVsplotchesTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "None";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < TVsplotchesTexture >;
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	AddressU  = Clamp;
	AddressV  = Clamp;
	MinFilter = Linear;
	MagFilter = Linear;
	MipFilter = None;
	SET_NO_ANISOTROPY
};


texture normalNoiseTexture : TEXTURE;
sampler normalNoiseInput : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="normalNoiseTexture"; 
	string MinFilter = "Point";
	string MagFilter = "Point";
	string MipFilter = "None";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < normalNoiseTexture >;
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	AddressU  = Clamp;
	AddressV  = Clamp;
	MinFilter = Point;
	MagFilter = Point;
	MipFilter = None;
	SET_NO_ANISOTROPY
};

texture3D random3DTexture : TEXTURE;
sampler3D random3DInput : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="random3DTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "None";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	string AddressW  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < random3DTexture >;
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	AddressU  = Wrap;
	AddressV  = Wrap;
	AddressW  = Wrap;
	MinFilter = Linear;
	MagFilter = Linear;
	MipFilter = None;
	SET_NO_ANISOTROPY
};


float	time_0_1;
float	time_0_X;

float	frameSharpness = 3.4f;
float	frameShape = 0.24f;
float	frameLimit = 0.32f;
float	frameIntensity = 1.0f;

float3	TVFrame( float3 c, float2 uv )
{
	// TV frame shape
	float2	pos = (uv.xy*2.0f)-1.0f;
	float f = (1 - pos.x * pos.x) * (1 - pos.y * pos.y);
	float frame = saturate(frameSharpness * (pow(f, frameShape) - frameLimit));

#ifdef USE_ALPHA_BLEND_ZERO_SRCCOLOR
	c.rgb = 1.0f - (frameIntensity * (1.0f-frame));	// this gets multiplied by the dest colour
	c.rgb = lerp( float3(1.0f,1.0f,1.0f), c.rgb, alpha_output );
#else
	float3	result = c.rgb * frame;
	c.rgb = lerp( c.rgb, result, frameIntensity );
#endif

	return c;
}

float	FlickerSpeed = 1.65f;

float3	TVFlicker( float3 c, float2 uv )
{
	float3	result = c.rgb;

	float darken = frac(1500.0f*time_0_1*FlickerSpeed);		// speed
	// we want darken to cycle between 0.8 and 1.0
	darken = (darken*0.2f)+1.05f;	// actually we LIGHTEN the scene a little, to make up for all the darkening we've being doing with the noise
	result = result * darken;

	c.rgb = result;

	return c;
}

float	InterferenceIntensity = 0.15f;

float3	TVInterference( float3 c, float2 uv )
{
	float3	result = c.rgb;

	float2	pos = (uv.xy*2.0f)-1.0f;
	float rand = tex3D( random3DInput, float3(4.0f * pos, time_0_X*0.25f) ).r - 0.2f;
	rand *= InterferenceIntensity;
	rand += (1.0f - InterferenceIntensity);
	result = result * rand;

	c.rgb = result;

	return c;
}

float3	TVFlickerAndInterference( float3 c, float2 uv )
{
	float3	result = c.rgb;

	float darken = frac(1500.0f*time_0_1*FlickerSpeed);		// speed
	// we want darken to cycle between 0.8 and 1.0
	darken = (darken*0.2f)+1.05f;	// actually we LIGHTEN the scene a little, to make up for all the darkening we've being doing with the noise
	result = result * darken;

	float2	pos = (uv.xy*2.0f)-1.0f;
	float rand = tex3D( random3DInput, float3(4.0f * pos, time_0_X*0.25f) ).r - 0.2f;
	rand *= InterferenceIntensity;
	rand += (1.0f - InterferenceIntensity);
	result = result * rand;

	c.rgb = result;

	return c;
}


float	BlotchIntensity = 0.15f;
float	BlotchFrequency = 0.005f;

float3	TVBlotches( float3 c, float2 uv )
{
	float3	result = c.rgb;

	// get 2 random vectors
	float	stime = frac( time_0_X*BlotchFrequency );
	float4 randVec0 = tex1D( noise1DInput, stime );
	float4 randVec1 = tex1D( noise1DInput2, stime );

	const float DirtFrequency = 35.07f;
	
	// scale address texture
	float sizeScale = ((randVec0.r + randVec0.g + randVec0.b + randVec0.a) / 4.0f) * 2.0f * 1.0f;
	uv = frac((uv + randVec1.xy) / float2(DirtFrequency * (1.0f+sizeScale), 200.0f));

	// normalized sub-texel coords
	float2 subTexelCoord0 = frac(uv.xy * 1024.0f);
	subTexelCoord0.y = 1.0f - subTexelCoord0.y;

	// get sprite address   
	float3 spriteCoords0 = tex2D( normalNoiseInput, uv.xy );
	
	// get sprite address into paged texture coords space
	spriteCoords0.xy = (spriteCoords0.xy * 8.0f);
	spriteCoords0.xy = (spriteCoords0.xy - frac(spriteCoords0.xy)) / 8.0f;
	spriteCoords0.xy = spriteCoords0.xy + (subTexelCoord0 * (1.0f/8.0f));

	float3	splotch = tex2D(TVsplotchesInput, spriteCoords0.xy);
	splotch *= BlotchIntensity;
	splotch += (1.0f-BlotchIntensity);
	result *= splotch;

	c.rgb = result;

	return c;
}

float	TVSepiaAmount = 0.125f;

float3	TVEffect( float3 c, float2 uv )
{
	float3	result = c.rgb;

	// TV frame shape
	float2	pos = (uv.xy*2.0f)-1.0f;
	float f = (1 - pos.x * pos.x) * (1 - pos.y * pos.y);
	float frame = saturate(frameSharpness * (pow(f, frameShape) - frameLimit));
	result = lerp( result, result * frame, frameIntensity );

	// Interference ... just a texture filled with rand()
	float rand = tex3D( random3DInput, float3(4.0f * pos, time_0_X*0.25f) ).r - 0.2f;
	rand *= InterferenceIntensity;
	rand += (1.0f - InterferenceIntensity);
	result = result * rand;

	// get 2 random vectors
	float	stime = frac( time_0_X*BlotchFrequency );
	float4 randVec0 = tex1D( noise1DInput, stime );
	float4 randVec1 = tex1D( noise1DInput2, stime );

	const float DirtFrequency = 35.07f;
	
	// scale address texture
	float sizeScale = ((randVec0.r + randVec0.g + randVec0.b + randVec0.a) / 4.0f) * 2.0f * 1.0f;
	uv = frac((uv + randVec1.xy) / float2(DirtFrequency * (1.0f+sizeScale), 200.0f));
	
	// normalized sub-texel coords
	float2 subTexelCoord0 = frac(uv.xy * 1024.0f);
	subTexelCoord0.y = 1.0f - subTexelCoord0.y;
	
	// get sprite address   
	float3 spriteCoords0 = tex2D( normalNoiseInput, uv.xy );
	
	// get sprite address into paged texture coords space
	spriteCoords0.xy = (spriteCoords0.xy * 8.0f);
	spriteCoords0.xy = (spriteCoords0.xy - frac(spriteCoords0.xy)) / 8.0f;
	spriteCoords0.xy = spriteCoords0.xy + (subTexelCoord0 * (1.0f/8.0f));

	float3	splotch = tex2D(TVsplotchesInput, spriteCoords0.xy);
	splotch *= BlotchIntensity;
	splotch += (1.0f-BlotchIntensity);
	result *= splotch;


	// do slight flicker
	float darken = frac(1500.0f*time_0_1*FlickerSpeed);		// speed
	// we want darken to cycle between 0.8 and 1.0
	darken = (darken*0.2f)+1.05f;	// actually we LIGHTEN the scene a little, to make up for all the darkening we've being doing with the noise
	result = result * darken;


	// small amount of sepia
	{
		c.rgb = result;
		float3 LightColour = float3( 1.0f, 0.9f, 0.5f );
		float3 DarkColor = float3( 0.2f, 0.05f, 0.0f );
		float Desat = 0.5f;
	  float3 scnColor = LightColour * c.rgb;
	  float gray = dot( float3(0.3f, 0.59f, 0.11f),scnColor);
	  float3 muted = lerp(scnColor,gray.rrr,Desat);
	  float3 sepia = lerp(DarkColor,LightColour,gray);
	  result = lerp(muted,sepia,TVSepiaAmount);
	}

	c.rgb = result;

	return c;
}

float	mosaic_tilesize = 8.0f;

float3	DoMosaic( float3 c_in, float2 uv )
{
	float2	wh = texinput_width_height;

	float	tilesize = mosaic_tilesize;
	bool	edge = false;

	float2	xy = uv*wh;
	xy /= tilesize;
	xy = floor( xy );

	float2	mod = ((uv*wh)/tilesize) - xy;
	//if( mod.x<0.1f || mod.y<0.1f )
	if( mod.x<1.0f/tilesize || mod.y<1.0f/tilesize )
	{
		edge = true;
	}

	xy *= tilesize;
	xy += tilesize/2.0f;
	uv = xy/wh;

	float3	c = SCENE_TEX2D( uv ).rgb;

	c *= 256.0f;
	c /= 16.0f;
	c = floor( c );
	c *= 16.0f;
	c += 8.0f;
	c /= 256.0f;

	if( edge )
	{
		c *= 0.6f;
	}
	else
	{
		c = lerp( c_in, c, 0.9f );
	}

	return c;
}



float2	impact_direction1 = float2( 1.0f, 1.0f );
float2	impact_direction2 = float2( -1.0f, -1.0f );

float3	Impact( float3 c_in, float2 uv )
{
	const float3 impact_colour = float3( 0.4f, 0.6f,0.6f );

	float2	dir = (uv*2.0f)-1.0f;
	float		distsq = dot(dir,dir);///1.414214f;
	float		intensity = distsq;
	
	float		control_texture = 1.0f;//tex2D( impactInput, uv ).r;
	intensity *= (control_texture*0.6f)+0.4f;

	float3	sample = c_in;
	
	float	directional_amount = saturate( max( dot( dir, impact_direction1 ), dot( dir, impact_direction2 ) ) );
	directional_amount = 1.0f-directional_amount;
	directional_amount *= directional_amount;
	directional_amount *= directional_amount;
	directional_amount = 1.0f-directional_amount;

	const float	radius = 0.60f;
	const float	radius2 = 0.95f;
	float	within_two_radius = saturate( (sqrt(distsq)-radius) / (radius2-radius) );
	if( within_two_radius>0.0f )
	{
		float	r = 1.0f-((1.0f-within_two_radius)*(1.0f-within_two_radius));
		float	sample_radius = lerp( radius, radius2, r*0.25f );
		uv = normalize(dir)*sample_radius;
		uv = (uv+1.0f)*0.5f;
		float	alpha = within_two_radius;
		alpha = 1.0f-alpha;
		alpha *= alpha;
		alpha *= alpha;
		alpha = 1.0f-alpha;
		sample = lerp( sample, SCENE_TEX2D( uv ).rgb, alpha*0.9f );
	}
	
	float3	temp = sample*float3(1.0f,0.75f,1.0f);
	temp = saturate( (temp-float3(0.25f,0.0f,0.0f))*2.0f );
	temp = saturate(temp-float3(0.4f, 0.4f, 0.4f));
	temp *= 8.0f;
	sample = lerp( sample, temp, within_two_radius*0.15f );

	float3	colourised = lerp( float3(1.0f,1.0f,1.0f), impact_colour, saturate(distsq*(((1.0f-control_texture)*0.2f)+0.8f)) );
	sample *= colourised;

	directional_amount *= 0.4f;	// Reduce affect of filter a little
	sample = lerp( c_in, sample, directional_amount );
	
	return sample;
}


float	blur_pixels = 1.0f;

float3	Blur1( float3 c_in, float2 uv )
{
	float2	wh = blur_pixels / texinput_width_height;

	float3	c = c_in;
	c += SCENE_TEX2D( uv+float2(-wh.x,-wh.y) ).rgb;
	c += SCENE_TEX2D( uv+float2(0.0f,-wh.y) ).rgb;
	c += SCENE_TEX2D( uv+float2(wh.x,-wh.y) ).rgb;
	c += SCENE_TEX2D( uv+float2(-wh.x,0.0f) ).rgb;
	c += SCENE_TEX2D( uv+float2(wh.x,0.0f) ).rgb;
	c += SCENE_TEX2D( uv+float2(-wh.x,wh.y) ).rgb;
	c += SCENE_TEX2D( uv+float2(0.0f,wh.y) ).rgb;
	c += SCENE_TEX2D( uv+float2(wh.x,wh.y) ).rgb;

	c /= 9.0f;

	return c;
}

float3	Blur1AndGreyscale( float3 c_in, float2 uv )
{
	float2	wh = blur_pixels / texinput_width_height;

	float3	c = c_in;
	c += SCENE_TEX2D( uv+float2(-wh.x,-wh.y) ).rgb;
	c += SCENE_TEX2D( uv+float2(0.0f,-wh.y) ).rgb;
	c += SCENE_TEX2D( uv+float2(wh.x,-wh.y) ).rgb;
	c += SCENE_TEX2D( uv+float2(-wh.x,0.0f) ).rgb;
	c += SCENE_TEX2D( uv+float2(wh.x,0.0f) ).rgb;
	c += SCENE_TEX2D( uv+float2(-wh.x,wh.y) ).rgb;
	c += SCENE_TEX2D( uv+float2(0.0f,wh.y) ).rgb;
	c += SCENE_TEX2D( uv+float2(wh.x,wh.y) ).rgb;

	c /= 9.0f;

	float3	grey = dot( c.rgb, float3(0.3f, 0.59f, 0.11f) );
	c.rgb = lerp( c.rgb, grey, greyscale_amount );

	return c;
}

float3	Blur1AndGreyscaleAndContrastIncrease( float3 c_in, float2 uv )
{
	float2	wh = blur_pixels / texinput_width_height;

	float3	c = c_in;
	c += SCENE_TEX2D( uv+float2(-wh.x,-wh.y) ).rgb;
	c += SCENE_TEX2D( uv+float2(0.0f,-wh.y) ).rgb;
	c += SCENE_TEX2D( uv+float2(wh.x,-wh.y) ).rgb;
	c += SCENE_TEX2D( uv+float2(-wh.x,0.0f) ).rgb;
	c += SCENE_TEX2D( uv+float2(wh.x,0.0f) ).rgb;
	c += SCENE_TEX2D( uv+float2(-wh.x,wh.y) ).rgb;
	c += SCENE_TEX2D( uv+float2(0.0f,wh.y) ).rgb;
	c += SCENE_TEX2D( uv+float2(wh.x,wh.y) ).rgb;

	c /= 9.0f;

	float3	grey = dot( c.rgb, float3(0.3f, 0.59f, 0.11f) );
	c.rgb = lerp( c.rgb, grey, greyscale_amount );

	c = ContrastIncrease( c );

	return c;
}


float3	Blur2( float3 c_in, float2 uv )
{
	float2	wh = blur_pixels / texinput_width_height;

	float3	c = c_in;
	c += SCENE_TEX2D( uv+float2(0.0f,-wh.y*2.0f) ).rgb;
	c += SCENE_TEX2D( uv+float2(-wh.x,-wh.y) ).rgb;
	c += SCENE_TEX2D( uv+float2(0.0f,-wh.y) ).rgb;
	c += SCENE_TEX2D( uv+float2(wh.x,-wh.y) ).rgb;
	c += SCENE_TEX2D( uv+float2(-wh.x*2.0f,0.0f) ).rgb;
	c += SCENE_TEX2D( uv+float2(-wh.x,0.0f) ).rgb;
	c += SCENE_TEX2D( uv+float2(wh.x,0.0f) ).rgb;
	c += SCENE_TEX2D( uv+float2(wh.x*2.0f,0.0f) ).rgb;
	c += SCENE_TEX2D( uv+float2(-wh.x,wh.y) ).rgb;
	c += SCENE_TEX2D( uv+float2(0.0f,wh.y) ).rgb;
	c += SCENE_TEX2D( uv+float2(wh.x,wh.y) ).rgb;
	c += SCENE_TEX2D( uv+float2(0.0f,wh.y*2.0f) ).rgb;

	c /= 13.0f;

	return c;
}

float3	multiply_RGB = float3( 1.0f, 1.0f, 1.0f );

float3	MultiplyRGB( float3 c )
{
#ifdef USE_ALPHA_BLEND_ZERO_SRCCOLOR
	c = multiply_RGB;
	c.rgb = lerp( float3(1.0f,1.0f,1.0f), c.rgb, alpha_output );
#else
	c *= multiply_RGB;
#endif
	return c;
}



float3	CalcFilter( float3 c, float2 texUV )
{
	float3	original_c = c;

	#if ( POST_PROCESS_EFFECT_INDEX==NONE )
	{
		#ifdef USE_ALPHA_BLEND_ZERO_SRCCOLOR
			c = float3( 1.0f, 1.0f, 1.0f );	// just do blend with target so the output does not change
			return c;
		#else
			return c;
		#endif
	}
	#elif ( POST_PROCESS_EFFECT_INDEX==OLD_TV )
	{
		c = TVEffect( c, texUV );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX==TV_FRAME )
	{
		c = TVFrame( c, texUV );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX==TV_FLICKER )
	{
		c = TVFlicker( c, texUV );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX==TV_INTERFERENCE )
	{
		c = TVInterference( c, texUV );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX==TV_BLOTCHES )
	{
		c = TVBlotches( c, texUV );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX==TV_FLICKER_AND_INTERFERENCE )
	{
		c = TVFlickerAndInterference( c, texUV );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX==GREYSCALE )
	{
		c = GreyScale( c );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX==GREYSCALE_RGB )
	{
		c = GreyScaleRGB( c );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX==SEPIA )
	{
		c = Sepia( c );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX==COLOURISED )
	{
		c = Colourised( c );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX==SATURATE_COLOURISE )
	{
		c = SaturateColourise( c );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX==CONTRAST_INCREASE )
	{
		c = ContrastIncrease( c );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX==REMOVE_COLOUR )
	{
		c = RemoveColour( c );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX==SATURATE )
	{
		c = Saturate( c );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX==MOSAIC )
	{
		c = DoMosaic( c, texUV );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX==BLUR1 )
	{
		c = Blur1( c, texUV );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX==BLUR2 )
	{
		c = Blur2( c, texUV );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX==MULTIPLY_RGB )
	{
		c = MultiplyRGB( c );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX==BLUR1_AND_GREYSCALE )
	{
		c = Blur1AndGreyscale( c, texUV );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX==BLUR1_AND_GREYSCALE_AND_CONTRAST_INCREASE )
	{
		c = Blur1AndGreyscaleAndContrastIncrease( c, texUV );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX==IMPACT )
	{
		c = Impact( c, texUV );
	}
	#endif

	#if ( POST_PROCESS_EFFECT_INDEX2==NONE )
	{
	}
	#elif ( POST_PROCESS_EFFECT_INDEX2==OLD_TV )
	{
		c = TVEffect( c, texUV );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX2==TV_FRAME )
	{
		c = TVFrame( c, texUV );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX2==TV_FLICKER )
	{
		c = TVFlicker( c, texUV );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX2==TV_INTERFERENCE )
	{
		c = TVInterference( c, texUV );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX2==TV_BLOTCHES )
	{
		c = TVBlotches( c, texUV );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX2==TV_FLICKER_AND_INTERFERENCE )
	{
		c = TVFlickerAndInterference( c, texUV );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX2==GREYSCALE )
	{
		c = GreyScale( c );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX2==GREYSCALE_RGB )
	{
		c = GreyScaleRGB( c );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX2==SEPIA )
	{
		c = Sepia( c );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX2==COLOURISED )
	{
		c = Colourised( c );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX2==SATURATE_COLOURISE )
	{
		c = SaturateColourise( c );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX2==CONTRAST_INCREASE )
	{
		c = ContrastIncrease( c );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX2==REMOVE_COLOUR )
	{
		c = RemoveColour( c );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX2==SATURATE )
	{
		c = Saturate( c );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX2==MOSAIC )
	{
		c = DoMosaic( c, texUV );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX2==BLUR1 )
	{
		c = Blur1( c, texUV );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX2==BLUR2 )
	{
		c = Blur2( c, texUV );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX2==MULTIPLY_RGB )
	{
		c = MultiplyRGB( c );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX2==BLUR1_AND_GREYSCALE )
	{
		c = Blur1AndGreyscale( c, texUV );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX2==BLUR1_AND_GREYSCALE_AND_CONTRAST_INCREASE )
	{
		c = Blur1AndGreyscaleAndContrastIncrease( c, texUV );
	}
	#endif

	#if ( POST_PROCESS_EFFECT_INDEX3==NONE )
	{
	}
	#elif ( POST_PROCESS_EFFECT_INDEX3==OLD_TV )
	{
		c = TVEffect( c, texUV );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX3==TV_FRAME )
	{
		c = TVFrame( c, texUV );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX3==TV_FLICKER )
	{
		c = TVFlicker( c, texUV );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX3==TV_INTERFERENCE )
	{
		c = TVInterference( c, texUV );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX3==TV_BLOTCHES )
	{
		c = TVBlotches( c, texUV );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX3==TV_FLICKER_AND_INTERFERENCE )
	{
		c = TVFlickerAndInterference( c, texUV );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX3==GREYSCALE )
	{
		c = GreyScale( c );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX2==GREYSCALE_RGB )
	{
		c = GreyScaleRGB( c );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX3==SEPIA )
	{
		c = Sepia( c );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX3==COLOURISED )
	{
		c = Colourised( c );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX3==SATURATE_COLOURISE )
	{
		c = SaturateColourise( c );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX3==CONTRAST_INCREASE )
	{
		c = ContrastIncrease( c );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX3==REMOVE_COLOUR )
	{
		c = RemoveColour( c );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX3==SATURATE )
	{
		c = Saturate( c );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX3==MOSAIC )
	{
		c = DoMosaic( c, texUV );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX3==BLUR1 )
	{
		c = Blur1( c, texUV );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX3==BLUR2 )
	{
		c = Blur2( c, texUV );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX3==MULTIPLY_RGB )
	{
		c = MultiplyRGB( c );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX3==BLUR1_AND_GREYSCALE )
	{
		c = Blur1AndGreyscale( c, texUV );
	}
	#elif ( POST_PROCESS_EFFECT_INDEX3==BLUR1_AND_GREYSCALE_AND_CONTRAST_INCREASE )
	{
		c = Blur1AndGreyscaleAndContrastIncrease( c, texUV );
	}
	#endif

	return c;
}

float4	DoSimplePostProcessingEffect( float3 colour_in, float2 texUV )
{
	float4	c;

	c.rgb = CalcFilter( colour_in.rgb, texUV );
#ifdef USE_ALPHA_BLEND_ZERO_SRCCOLOR
	c.a = 1.0f;
#else
	c.a = alpha_output;
#endif

	return c;
}
